function T_wire(P)
% T_wire — 3D plot of the T-wire structure (ONLY STANDARD CELLS, no cuts)
%  - 3 arms: North (+Y), South (-Y), East (+X); West side filled by fillers
%  - 4 standard cells per arm (pitch = W)
%  - Central STANDARD SQUARE block (L x L) with full-vacuum slab + Au cap + Ti rim
%
% Optional input:
%   P : struct with fields overriding defaults (units: nm)
%
% Example:
%   T_wire;                                   % plot with defaults
%   P = struct('L',30,'W',12,'origin',[0 0 0]); T_wire(P);

%% ---------- Defaults ----------
Pd = struct( ...
    'L',24, 'W',10, 'H',2, ...                 % SiO2 base (Lx, Wy, Hz)
    'origin',[0 0 0], ...                      % [x0 y0 z0] (nm)
    'H_Ti_bottom',1, ...                       % Ti bottom (height only)
    'H_gold_bottom',2, ...                     % Au bottom (height only)
    'L_vac',4, 'H_vac',3, ...                  % trench (Lx, Hz)
    'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ... % top gold (Lx, Wy, Hz)
    'T_film',0.5, ...                          % Ti rim thickness
    'CapSide_center',22 ...                    % Au cap side (central block)
);
N = 4;                                         % 4 standard cells per arm

if nargin<1 || isempty(P), P = Pd; else, P = mergeStruct(Pd,P); end

%% ---------- Build geometry ----------
x0 = P.origin(1); y0 = P.origin(2); z0 = P.origin(3);
pitch = P.W;                                   % ONLY standard cells -> pitch = W

% Total height for axis box
H_tot = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;

% ===== North arm (along +Y), 4 STANDARD cells
y_cursor = y0;
GN1 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;
GN2 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;
GN3 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;
GN4 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;

% ===== Central standard square block (L x L)
yB = y_cursor;
Gc = buildCentralSquareCellVerts(P, [x0 yB z0]);

% ===== South arm (along -Y), 4 STANDARD cells
y_cursor = yB + P.L;
GS4 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;
GS3 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;
GS2 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;
GS1 = buildCellVertsY(P, [x0 y_cursor z0]); y_cursor = y_cursor + pitch;

% ===== East arm (along +X), 4 STANDARD cells
xB2 = x0 + P.L;
x_cursor = xB2;
GE4 = buildCellVertsX(P, [x_cursor yB z0], P.W, P.L); x_cursor = x_cursor + pitch;
GE3 = buildCellVertsX(P, [x_cursor yB z0], P.W, P.L); x_cursor = x_cursor + pitch;
GE2 = buildCellVertsX(P, [x_cursor yB z0], P.W, P.L); x_cursor = x_cursor + pitch;
GE1 = buildCellVertsX(P, [x_cursor yB z0], P.W, P.L); x_cursor = x_cursor + pitch;

% ===== Fillers (West side + corners)
arm_span = N * pitch;                           % all standard -> span = 4*W
yB2 = yB + P.L;
FF_NE = buildFillerLayers([xB2           yB2           z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
FF_NW = buildFillerLayers([x0 - arm_span yB2           z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
FF_SW = buildFillerLayers([x0 - arm_span yB - arm_span z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
FF_SE = buildFillerLayers([xB2           yB - arm_span z0], arm_span, arm_span, [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);
FX    = buildFillerLayers([x0 - arm_span yB            z0], arm_span, P.L,      [P.H + P.H_Ti_bottom, P.H_gold_bottom, P.H_vac, P.H_Au_top]);

%% ---------- Plot ----------
[c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw] = colorsAndStyle();
F = faces6();

fig = figure('Color','w','Position',[100 80 1180 740]);
ax  = axes('Parent',fig); hold(ax,'on'); view(ax,3); axis(ax,'vis3d');
grid(ax,'on'); box(ax,'on'); ax.LineWidth=0.8; ax.GridAlpha=0.12; ax.MinorGridAlpha=0.06;

% North arm
drawGeom(ax, GN1, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, true);
drawGeom(ax, GN2, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GN3, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GN4, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% Center block
drawGeom(ax, Gc,  c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% South arm
drawGeom(ax, GS4, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GS3, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GS2, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GS1, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% East arm
drawGeom(ax, GE4, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GE3, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GE2, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax, GE1, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% Fillers
drawFillerLayers(ax, FF_NE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, 'Filler', false);
drawFillerLayers(ax, FF_NW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FF_SW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FF_SE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FX,    c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);

% Axes, labels, limits
axis(ax,'equal'); view(ax,35,25);
xlabel(ax,'X [nm]','Interpreter','tex'); ylabel(ax,'Y [nm]','Interpreter','tex'); zlabel(ax,'Z [nm]','Interpreter','tex');
title(ax, sprintf('T-wire (ALL STANDARD CELLS) — N=%d, pitch=%.3g nm', N, pitch));

Y_span = arm_span + P.L + arm_span;  % north arm + center + south arm
X_span = P.L + arm_span;             % center + east arm
marg = 0.12 * [X_span, Y_span, H_tot];

xlim(ax,[x0 - arm_span - marg(1), x0 + P.L + arm_span + marg(1)]);
ylim(ax,[y0 - arm_span - marg(2), y0 + Y_span + marg(2)]);
zlim(ax,[z0 - marg(3), z0 + H_tot + marg(3)]);

plot3(ax,P.origin(1),P.origin(2),P.origin(3),'ko','MarkerFaceColor','k','HandleVisibility','off');
legend(ax,'Location','bestoutside');
hold(ax,'off');

end % ===== END MAIN =====

%% ========================== Helpers ==========================
function G = buildCellVertsY(P, o)
    % Standard cell oriented along +Y
    Wy = P.W; Lx = P.L;

    V_base = vbox(o, Lx, Wy, P.H);
    o_Ti = o + [0 0 P.H];
    V_Ti = vbox(o_Ti, Lx, Wy, P.H_Ti_bottom);
    o_Au = o + [0 0 (P.H + P.H_Ti_bottom)];
    V_Au = vbox(o_Au, Lx, Wy, P.H_gold_bottom);

    z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;

    % HfO2 walls + vacuum slot (centered)
    L_vac_eff = min(max(P.L_vac,0), Lx);
    L_left  = max(0, (Lx - L_vac_eff)/2);
    L_right = max(0, Lx - (L_left + L_vac_eff));
    V_left  = vbox([o(1)               o(2) z0_top], L_left,       Wy, P.H_vac);
    V_vac   = vbox([o(1)+L_left        o(2) z0_top], L_vac_eff,    Wy, P.H_vac);
    V_right = vbox([o(1)+L_left+L_vac_eff o(2) z0_top], L_right,   Wy, P.H_vac);

    % Top Au cap + Ti rim (centered)
    x0_cap = o(1) + (Lx - P.L_Au_top)/2;
    y0_cap = o(2) + (Wy - P.W_Au_top)/2;
    z0_cap = z0_top + P.H_vac;
    V_cap  = vbox([x0_cap, y0_cap, z0_cap], P.L_Au_top, P.W_Au_top, P.H_Au_top);

    V_film_xL = vbox([x0_cap - P.T_film, y0_cap, z0_cap], P.T_film, P.W_Au_top, P.H_Au_top);
    V_film_xR = vbox([x0_cap + P.L_Au_top, y0_cap, z0_cap], P.T_film, P.W_Au_top, P.H_Au_top);
    V_film_yF = vbox([x0_cap - P.T_film, y0_cap - P.T_film, z0_cap], P.L_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);
    V_film_yB = vbox([x0_cap - P.T_film, y0_cap + P.W_Au_top, z0_cap], P.L_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);

    % SiO2 frame around the rim (top layer)
    [V_frame_xL,V_frame_xR,V_frame_yF,V_frame_yB] = topSiO2Frame(o, z0_cap, Lx, Wy, x0_cap, y0_cap, P.L_Au_top, P.W_Au_top, P.T_film, P.H_Au_top);

    G.base = V_base; G.ti = V_Ti; G.au = V_Au;
    G.left = V_left; G.vac = V_vac; G.right= V_right;
    G.cap = V_cap; G.fxL = V_film_xL; G.fxR = V_film_xR;
    G.fyF = V_film_yF; G.fyB= V_film_yB;
    G.frmL = V_frame_xL; G.frmR = V_frame_xR;
    G.frmF = V_frame_yF; G.frmB = V_frame_yB;
end

function G = buildCellVertsX(P, o, Lx_cell, Ly_cell)
    % Standard cell oriented along +X
    if nargin < 3, Lx_cell = P.W; end
    if nargin < 4, Ly_cell = P.L; end

    V_base = vbox(o, Lx_cell, Ly_cell, P.H);
    o_Ti = o + [0 0 P.H];
    V_Ti = vbox(o_Ti, Lx_cell, Ly_cell, P.H_Ti_bottom);
    o_Au = o + [0 0 (P.H + P.H_Ti_bottom)];
    V_Au = vbox(o_Au, Lx_cell, Ly_cell, P.H_gold_bottom);

    z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;

    % HfO2 walls + vacuum slot (centered along Y)
    L_vac_eff = min(max(P.L_vac,0), Ly_cell);
    L_left  = max(0, (Ly_cell - L_vac_eff)/2);
    L_right = max(0, Ly_cell - (L_left + L_vac_eff));
    V_left  = vbox([o(1) o(2) z0_top], Lx_cell, L_left,  P.H_vac);
    V_vac   = vbox([o(1) o(2)+L_left z0_top], Lx_cell, L_vac_eff, P.H_vac);
    V_right = vbox([o(1) o(2)+L_left+L_vac_eff z0_top], Lx_cell, L_right, P.H_vac);

    % Top Au cap + Ti rim (centered along Y)
    x0_cap = o(1) + (Lx_cell - P.W_Au_top)/2;
    y0_cap = o(2) + (Ly_cell - P.L_Au_top)/2;
    z0_cap = z0_top + P.H_vac;
    V_cap  = vbox([x0_cap, y0_cap, z0_cap], P.W_Au_top, P.L_Au_top, P.H_Au_top);

    V_film_xL = vbox([x0_cap - P.T_film, y0_cap, z0_cap], P.T_film, P.L_Au_top, P.H_Au_top);
    V_film_xR = vbox([x0_cap + P.W_Au_top, y0_cap, z0_cap], P.T_film, P.L_Au_top, P.H_Au_top);
    V_film_yF = vbox([x0_cap - P.T_film, y0_cap - P.T_film, z0_cap], P.W_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);
    V_film_yB = vbox([x0_cap - P.T_film, y0_cap + P.L_Au_top, z0_cap], P.W_Au_top + 2*P.T_film, P.T_film, P.H_Au_top);

    % SiO2 frame on top
    [V_frame_xL,V_frame_xR,V_frame_yF,V_frame_yB] = topSiO2Frame(o, z0_cap, Lx_cell, Ly_cell, x0_cap, y0_cap, P.W_Au_top, P.L_Au_top, P.T_film, P.H_Au_top);

    G.base = V_base; G.ti = V_Ti; G.au = V_Au;
    G.left = V_left; G.vac = V_vac; G.right= V_right;
    G.cap = V_cap; G.fxL = V_film_xL; G.fxR = V_film_xR;
    G.fyF = V_film_yF; G.fyB= V_film_yB;
    G.frmL = V_frame_xL; G.frmR = V_frame_xR;
    G.frmF = V_frame_yF; G.frmB = V_frame_yB;
end

function [V_frame_xL,V_frame_xR,V_frame_yF,V_frame_yB] = topSiO2Frame(o, z0_cap, Lx, Wy, x0_cap, y0_cap, Lcap, Wcap, Tfilm, Htop)
    x_in_min = x0_cap - Tfilm;
    x_in_max = x0_cap + Lcap + Tfilm;
    y_in_min = y0_cap - Tfilm;
    y_in_max = y0_cap + Wcap + Tfilm;
    x_base_min = o(1); x_base_max = o(1) + Lx;
    y_base_min = o(2); y_base_max = o(2) + Wy;

    T_left_frame  = max(0, x_in_min - x_base_min);
    T_right_frame = max(0, x_base_max - x_in_max);
    T_front_frame = max(0, y_in_min - y_base_min);
    T_back_frame  = max(0, y_base_max - y_in_max);
    z0_frame = z0_cap; H_frame = Htop;

    if T_left_frame > 0
        V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame, Wy, H_frame);
    else
        V_frame_xL = [];
    end
    if T_right_frame > 0
        V_frame_xR = vbox([x_in_max, y_base_min, z0_frame], T_right_frame, Wy, H_frame);
    else
        V_frame_xR = [];
    end
    if T_front_frame > 0
        V_frame_yF = vbox([x_in_min, y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame);
    else
        V_frame_yF = [];
    end
    if T_back_frame > 0
        V_frame_yB = vbox([x_in_min, y_in_max, z0_frame], max(0,x_in_max-x_in_min), T_back_frame, H_frame);
    else
        V_frame_yB = [];
    end
end

function Gc = buildCentralSquareCellVerts(P, o)
    % Center: STANDARD SQUARE (side L), FULL vacuum slab, centered Au cap + Ti rim + SiO2 frame
    Lside = P.L; Wside = P.L;

    V_base = vbox(o, Lside, Wside, P.H);
    V_Ti   = vbox(o+[0 0 P.H], Lside, Wside, P.H_Ti_bottom);
    V_Au   = vbox(o+[0 0 (P.H + P.H_Ti_bottom)], Lside, Wside, P.H_gold_bottom);

    z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;

    % FULL vacuum
    V_vac = vbox([o(1) o(2) z0_top], Lside, Wside, P.H_vac);

    % Cap
    capSide = min(P.CapSide_center, Lside);
    x0_cap = o(1) + (Lside - capSide)/2;
    y0_cap = o(2) + (Wside - capSide)/2;
    z0_cap = z0_top + P.H_vac;
    V_cap  = vbox([x0_cap, y0_cap, z0_cap], capSide, capSide, P.H_Au_top);

    % Ti rim
    V_film_xL = vbox([x0_cap - P.T_film, y0_cap, z0_cap], P.T_film, capSide, P.H_Au_top);
    V_film_xR = vbox([x0_cap + capSide,  y0_cap, z0_cap], P.T_film, capSide, P.H_Au_top);
    V_film_yF = vbox([x0_cap - P.T_film, y0_cap - P.T_film, z0_cap], capSide + 2*P.T_film, P.T_film, P.H_Au_top);
    V_film_yB = vbox([x0_cap - P.T_film, y0_cap + capSide,  z0_cap], capSide + 2*P.T_film, P.T_film, P.H_Au_top);

    % SiO2 frame on top
    [V_frame_xL,V_frame_xR,V_frame_yF,V_frame_yB] = topSiO2Frame(o, z0_cap, Lside, Wside, x0_cap, y0_cap, capSide, capSide, P.T_film, P.H_Au_top);

    Gc.base = V_base; Gc.ti = V_Ti; Gc.au = V_Au;
    Gc.vac  = V_vac;
    Gc.cap = V_cap; Gc.fxL = V_film_xL; Gc.fxR = V_film_xR;
    Gc.fyF = V_film_yF; Gc.fyB = V_film_yB;
    Gc.frmL = V_frame_xL; Gc.frmR = V_frame_xR;
    Gc.frmF = V_frame_yF; Gc.frmB = V_frame_yB;
end

function FL = buildFillerLayers(o, Lx, Wy, hvec)
    % Four stacked layers
    z = o(3);
    FL.V1 = vbox([o(1) o(2) z], Lx, Wy, hvec(1)); z = z + hvec(1);
    FL.V2 = vbox([o(1) o(2) z], Lx, Wy, hvec(2)); z = z + hvec(2);
    FL.V3 = vbox([o(1) o(2) z], Lx, Wy, hvec(3)); z = z + hvec(3);
    FL.V4 = vbox([o(1) o(2) z], Lx, Wy, hvec(4));
end

function drawFillerLayers(ax, FL, c1, c2, c3, c4, edgeCol, lw, F, legendName, offLegend)
    if nargin<10 || isempty(legendName), legendName=''; end
    if nargin<11, offLegend=false; end
    if ~isempty(legendName) && ~offLegend
        patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, 'DisplayName',legendName);
    else
        patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
    end
    patch(ax,'Vertices',FL.V2,'Faces',F,'FaceColor',c2,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
    patch(ax,'Vertices',FL.V3,'Faces',F,'FaceColor',c3,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
    patch(ax,'Vertices',FL.V4,'Faces',F,'FaceColor',c4,'EdgeColor',edgeCol,'LineWidth',lw, 'HandleVisibility','off');
end

function [c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeColor,lineWidth] = colorsAndStyle()
    c_base = [0.55 0.35 0.20]; % SiO2
    c_ti   = [0.20 0.20 0.20]; % Ti
    c_au   = [1.00 0.84 0.00]; % Au
    c_vac  = [0.00 0.60 0.00]; % vacuum (visual aid)
    a_vac  = 0.35;
    c_hfo2 = [0.80 0.65 0.45]; % HfO2
    edgeColor = 'k'; lineWidth = 1.0;
end

function drawGeom(ax,G, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, addLegend)
    drawPatch(ax,G.base, c_base, edgeCol,lw, addLegend,'SiO2', 1, F);
    drawPatch(ax,G.ti,   c_ti,   edgeCol,lw, addLegend,'Ti',   1, F);
    drawPatch(ax,G.au,   c_au,   edgeCol,lw, addLegend,'Au',   1, F);

    if isfield(G,'left') && ~isempty(G.left)
        drawPatch(ax,G.left,  c_hfo2, edgeCol,lw, addLegend,'HfO2', 1, F);
        drawPatch(ax,G.right, c_hfo2, edgeCol,lw, false,'', 1, F);
    end
    if isfield(G,'vac') && ~isempty(G.vac)
        drawPatch(ax,G.vac,  c_vac, edgeCol,lw, addLegend,'vacuum', a_vac, F);
    end

    if isfield(G,'cap'), drawPatch(ax,G.cap, c_au, edgeCol,lw, false,'', 1, F); end
    fld = {'fxL','fxR','fyF','fyB'};
    for ii=1:numel(fld)
        if isfield(G,fld{ii}), drawPatch(ax,G.(fld{ii}), c_ti, edgeCol,lw, false,'', 1, F); end
    end
    fld = {'frmL','frmR','frmF','frmB'};
    for ii=1:numel(fld)
        if isfield(G,fld{ii}), drawPatch(ax,G.(fld{ii}), c_base, edgeCol,lw, false,'', 1, F); end
    end
end

function ph = drawPatch(axh,V,col,edgeCol,lw, addToLegend, name, faceAlpha, F)
    if isempty(V), ph = []; return; end
    if addToLegend && ~isempty(name)
        ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
            'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',name);
    else
        ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
            'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
    end
end

function V = vbox(o,Lx,Wy,Hz)
    V = [ o;
          o + [Lx 0 0];
          o + [0 Wy 0];
          o + [0 0 Hz];
          o + [Lx Wy 0];
          o + [Lx 0 Hz];
          o + [0 Wy Hz];
          o + [Lx Wy Hz] ];
end

function F = faces6()
    F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
end

function S = mergeStruct(S, T)
    fn = fieldnames(T);
    for k=1:numel(fn), S.(fn{k}) = T.(fn{k}); end
end
